﻿using System;
using System.Collections.Generic;
using System.Runtime.Serialization;

namespace VA.PPMS.CRM.Plugins.Data
{
    //
    // Summary:
    //     A response object from the Bing Maps REST Services.
    [DataContract]
    public class Response
    {
        public Response() { }

        //
        // Summary:
        //     Copyright information.
        [DataMember(Name = "copyright", EmitDefaultValue = false)]
        public string Copyright { get; set; }
        //
        // Summary:
        //     A URL that references a brand image to support contractual branding requirements.
        [DataMember(Name = "brandLogoUri", EmitDefaultValue = false)]
        public string BrandLogoUri { get; set; }
        //
        // Summary:
        //     The HTTP Status code for the request.
        [DataMember(Name = "statusCode", EmitDefaultValue = false)]
        public int StatusCode { get; set; }
        //
        // Summary:
        //     A description of the HTTP status code.
        [DataMember(Name = "statusDescription", EmitDefaultValue = false)]
        public string StatusDescription { get; set; }
        //
        // Summary:
        //     A status code that offers additional information about authentication success
        //     or failure. Will be one of the following values: ValidCredentials, InvalidCredentials,
        //     CredentialsExpired, NotAuthorized, NoCredentials, None
        [DataMember(Name = "authenticationResultCode", EmitDefaultValue = false)]
        public string AuthenticationResultCode { get; set; }
        //
        // Summary:
        //     A collection of error descriptions.
        [DataMember(Name = "errorDetails", EmitDefaultValue = false)]
        public string[] ErrorDetails { get; set; }
        //
        // Summary:
        //     A unique identifier for the request.
        [DataMember(Name = "traceId", EmitDefaultValue = false)]
        public string TraceId { get; set; }
        //
        // Summary:
        //     A collection of ResourceSet objects. A ResourceSet is a container of Resources
        //     returned by the request.
        [DataMember(Name = "resourceSets", EmitDefaultValue = false)]
        public ResourceSet[] ResourceSets { get; set; }
    }

    //
    // Summary:
    //     A ResourceSet is a container of Resources returned by the request.
    [DataContract]
    public class ResourceSet
    {
        public ResourceSet() { }

        //
        // Summary:
        //     An estimate of the total number of resources in the ResourceSet.
        [DataMember(Name = "estimatedTotal", EmitDefaultValue = false)]
        public long EstimatedTotal { get; set; }
        //
        // Summary:
        //     A collection of one or more resources. The resources that are returned depend
        //     on the request.
        [DataMember(Name = "resources", EmitDefaultValue = false)]
        public Resource[] Resources { get; set; }
    }

    //
    // Summary:
    //     A resource that is returned depending on the request.
    [DataContract]
    [KnownType(typeof(Location))]
    [KnownType(typeof(Route))]
    [KnownType(typeof(TrafficIncident))]
    [KnownType(typeof(ImageryMetadata))]
    [KnownType(typeof(ElevationData))]
    [KnownType(typeof(SeaLevelData))]
    [KnownType(typeof(CompressedPointList))]
    [KnownType(typeof(GeospatialEndpointResponse))]
    [KnownType(typeof(DistanceMatrix))]
    [KnownType(typeof(DistanceMatrixAsyncStatus))]
    public class Resource
    {
        public Resource() { }

        //
        // Summary:
        //     Bounding box of the response. Structure [South Latitude, West Longitude, North
        //     Latitude, East Longitude]
        [DataMember(Name = "bbox", EmitDefaultValue = false)]
        public double[] BoundingBox { get; set; }
        //
        // Summary:
        //     The type of resource returned.
        [DataMember(Name = "__type", EmitDefaultValue = false)]
        public string Type { get; set; }
    }

    //
    // Summary:
    //     A Location response object which is returned when geocoding or reverse geocoding.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class Location : Resource
    {
        public Location() { }

        //
        // Summary:
        //     The name of the resource.
        [DataMember(Name = "name", EmitDefaultValue = false)]
        public string Name { get; set; }
        //
        // Summary:
        //     The latitude and longitude coordinates of the location.
        [DataMember(Name = "point", EmitDefaultValue = false)]
        public Point Point { get; set; }
        //
        // Summary:
        //     The classification of the geographic entity returned, such as Address.
        [DataMember(Name = "entityType", EmitDefaultValue = false)]
        public string EntityType { get; set; }
        //
        // Summary:
        //     The postal address for the location.
        [DataMember(Name = "address", EmitDefaultValue = false)]
        public Address Address { get; set; }
        //
        // Summary:
        //     The level of confidence that the geocoded location result is a match. Can be
        //     High, Medium, Low.
        [DataMember(Name = "confidence", EmitDefaultValue = false)]
        public string Confidence { get; set; }
        //
        // Summary:
        //     One or more match code values that represent the geocoding level for each location
        //     in the response. Can be Good, Ambiguous, UpHierarchy.
        [DataMember(Name = "matchCodes", EmitDefaultValue = false)]
        public string[] MatchCodes { get; set; }
        //
        // Summary:
        //     A collection of geocoded points that differ in how they were calculated and their
        //     suggested use.
        [DataMember(Name = "geocodePoints", EmitDefaultValue = false)]
        public Point[] GeocodePoints { get; set; }
        //
        // Summary:
        //     A collection of parsed values that shows how a location query string was parsed
        //     into one or more of the following address values. AddressLine, Locality, AdminDistrict,
        //     AdminDistrict2, PostalCode, CountryRegion, Landmark
        [DataMember(Name = "queryParseValues", EmitDefaultValue = false)]
        public QueryParseValue[] QueryParseValues { get; set; }
    }

    //
    // Summary:
    //     A route response object.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class Route : Resource
    {
        public Route() { }

        //
        // Summary:
        //     A unique ID for the resource.
        [DataMember(Name = "id", EmitDefaultValue = false)]
        public string Id { get; set; }
        //
        // Summary:
        //     The unit used for distance.
        [DataMember(Name = "distanceUnit", EmitDefaultValue = false)]
        public string DistanceUnit { get; set; }
        //
        // Summary:
        //     The unit used for distance as an Enum.
        public DistanceUnitType DistanceUnitType { get; set; }
        //
        // Summary:
        //     The unit used for time of travel.
        [DataMember(Name = "durationUnit", EmitDefaultValue = false)]
        public string DurationUnit { get; set; }
        //
        // Summary:
        //     The unit used for time as an Enum.
        public TimeUnitType TimeUnitType { get; set; }
        //
        // Summary:
        //     The physical distance covered by the entire route. This value is not supported
        //     for the Transit travel mode.
        [DataMember(Name = "travelDistance", EmitDefaultValue = false)]
        public double TravelDistance { get; set; }
        //
        // Summary:
        //     The time that it takes, in seconds, to travel a corresponding TravelDistance.
        [DataMember(Name = "travelDuration", EmitDefaultValue = false)]
        public double TravelDuration { get; set; }
        //
        // Summary:
        //     The time that it takes, in seconds, to travel a corresponding TravelDistance
        //     with current traffic conditions. This value is always provided for the complete
        //     route and does not depend on the availability of traffic information.
        [DataMember(Name = "travelDurationTraffic", EmitDefaultValue = false)]
        public double TravelDurationTraffic { get; set; }
        //
        // Summary:
        //     A description of the traffic congestion.
        [DataMember(Name = "trafficCongestion", EmitDefaultValue = false)]
        public string TrafficCongestion { get; set; }
        //
        // Summary:
        //     Information about the traffic data used.
        [DataMember(Name = "trafficDataUsed", EmitDefaultValue = false)]
        public string TrafficDataUsed { get; set; }
        //
        // Summary:
        //     Information about a section of a route between two waypoints.
        [DataMember(Name = "routeLegs", EmitDefaultValue = false)]
        public RouteLeg[] RouteLegs { get; set; }
        //
        // Summary:
        //     A representation of the path of a route. A RoutePath is defined by a Line element
        //     that contains of a collection of latitude and longitude points. The path of the
        //     route is the line that connects these points.
        [DataMember(Name = "routePath", EmitDefaultValue = false)]
        public RoutePath RoutePath { get; set; }
    }

    public enum DistanceUnitType
    {
        //
        // Summary:
        //     Distances in Kilometers.
        Kilometers = 0,
        //
        // Summary:
        //     Distances in Miles
        Miles = 1
    }

    public enum TimeUnitType
    {
        //
        // Summary:
        //     Time is in seconds.
        Seconds = 0,
        //
        // Summary:
        //     Time is in minutes.
        Minutes = 1
    }

    [DataContract]
    public class RouteLeg
    {
        public RouteLeg() { }

        //
        // Summary:
        //     The physical distance covered by a route leg.
        [DataMember(Name = "travelDistance", EmitDefaultValue = false)]
        public double TravelDistance { get; set; }
        //
        // Summary:
        //     The time that it takes, in seconds, to travel a corresponding TravelDistance.
        [DataMember(Name = "travelDuration", EmitDefaultValue = false)]
        public double TravelDuration { get; set; }
        //
        // Summary:
        //     The cost of the journey. Provided for transit routes in some countries.
        [DataMember(Name = "cost", EmitDefaultValue = false)]
        public double Cost { get; set; }
        //
        // Summary:
        //     A short description of the route.
        [DataMember(Name = "description", EmitDefaultValue = false)]
        public string Description { get; set; }
        //
        // Summary:
        //     The Point (latitude and longitude) that was used as the actual starting location
        //     for the route leg. In most cases, the ActualStart is the same as the requested
        //     waypoint. However, if a waypoint is not close to a road, the Routes API chooses
        //     a location on the nearest road as the starting point of the route. This ActualStart
        //     element contains the latitude and longitude of this new location.
        [DataMember(Name = "actualStart", EmitDefaultValue = false)]
        public Point ActualStart { get; set; }
        //
        // Summary:
        //     The Point (latitude and longitude) that was used as the actual ending location
        //     for the route leg. In most cases, the ActualEnd is the same as the requested
        //     waypoint. However, if a waypoint is not close to a road, the Routes API chooses
        //     a location on the nearest road as the ending point of the route. This ActualEnd
        //     element contains the latitude and longitude of this new location.
        [DataMember(Name = "actualEnd", EmitDefaultValue = false)]
        public Point ActualEnd { get; set; }
        //
        // Summary:
        //     Information about the location of the starting waypoint for a route. A StartLocation
        //     is provided only when the waypoint is specified as a landmark or an address.
        [DataMember(Name = "startLocation", EmitDefaultValue = false)]
        public Location StartLocation { get; set; }
        //
        // Summary:
        //     Information about the location of the endinpoint for a route. An EndLocation
        //     is provided only when the waypoint is specified as a landmark or an address.
        [DataMember(Name = "endLocation", EmitDefaultValue = false)]
        public Location EndLocation { get; set; }
        //
        // Summary:
        //     Information that defines a step in the route.
        [DataMember(Name = "itineraryItems", EmitDefaultValue = false)]
        public ItineraryItem[] ItineraryItems { get; set; }
        //
        // Summary:
        //     Information about a segments of the route leg defined by the route leg waypoints
        //     and any intermediate via-waypoints. For example, if the route leg has two via-waypoints
        //     in addition to start and end waypoints, there would be three (3) route sub-legs.
        [DataMember(Name = "routeSubLegs", EmitDefaultValue = false)]
        public RouteSubLeg[] RouteSubLegs { get; set; }
        //
        // Summary:
        //     Used for transit route responses. Shows the starting time for the starting point
        //     of the route. This tells you when to be at the starting waypoint depending on
        //     what you select as the dateTime and the timeType.
        [DataMember(Name = "startTime", EmitDefaultValue = false)]
        public string StartTime { get; set; }
        //
        // Summary:
        //     Used for transit route responses. Shows the starting time for the starting point
        //     of the route. This tells you when to be at the starting waypoint depending on
        //     what you select as the dateTime and the timeType.
        public DateTime StartTimeUtc { get; set; }
        //
        // Summary:
        //     Used for transit route responses. Shows the time of arrival when specific route
        //     is taken. This tells you when to be at the ending waypoint depending on what
        //     you select as the dateTime and the timeType parameters
        [DataMember(Name = "endTime", EmitDefaultValue = false)]
        public string EndTime { get; set; }
        //
        // Summary:
        //     Used for transit route responses. Shows the time of arrival when specific route
        //     is taken. This tells you when to be at the ending waypoint depending on what
        //     you select as the dateTime and the timeType parameters
        public DateTime EndTimeUtc { get; set; }
    }

    //
    // Summary:
    //     A representation of the path of a route. A RoutePath is defined by a Line element
    //     that contains of a collection of latitude and longitude points. The path of the
    //     route is the line that connects these points.
    [DataContract]
    public class RoutePath
    {
        public RoutePath() { }

        //
        // Summary:
        //     The route path coordinate information.
        [DataMember(Name = "line", EmitDefaultValue = false)]
        public Line Line { get; set; }
        //
        // Summary:
        //     Information on generalized coordinates to use in the route path for different
        //     resolutions with different zoom levels in a map.
        [DataMember(Name = "generalizations", EmitDefaultValue = false)]
        public Generalization[] Generalizations { get; set; }
    }

    [DataContract]
    public class Line
    {
        public Line() { }

        //
        // Summary:
        //     The type information of the object.
        [DataMember(Name = "type", EmitDefaultValue = false)]
        public string Type { get; set; }
        //
        // Summary:
        //     The coordinates that make up the route path.
        [DataMember(Name = "coordinates", EmitDefaultValue = false)]
        public double[][] Coordinates { get; set; }
    }

    [DataContract]
    public class Generalization
    {
        public Generalization() { }

        //
        // Summary:
        //     Specifies a subset of points from the complete set of route path points (line
        //     collection) to include in the RoutePathGeneralization. The list of point values
        //     in the route path line collection are given indices starting with 0. For example,
        //     an index value of 0 indicates that the first point in the Line collection is
        //     included in this RoutePathGeneralization.
        [DataMember(Name = "pathIndices", EmitDefaultValue = false)]
        public int[] PathIndices { get; set; }
        //
        // Summary:
        //     The tolerance specified in the route request that is associated with the RoutePathGeneralization.
        [DataMember(Name = "latLongTolerance", EmitDefaultValue = false)]
        public double LatLongTolerance { get; set; }
    }

    [DataContract]
    public class Point : Shape
    {
        public Point() { }

        //
        // Summary:
        //     The type information of the point.
        [DataMember(Name = "type", EmitDefaultValue = false)]
        public string Type { get; set; }
        //
        // Summary:
        //     The coordinate information of the point [Latitude,Longitude].
        [DataMember(Name = "coordinates", EmitDefaultValue = false)]
        public double[] Coordinates { get; set; }
        //
        // Summary:
        //     The method that was used to compute the geocode point. Can be one of the following
        //     values: Interpolation: The geocode point was matched to a point on a road using
        //     interpolation. InterpolationOffset: The geocode point was matched to a point
        //     on a road using interpolation with an additional offset to shift the point to
        //     the side of the street. Parcel: The geocode point was matched to the center of
        //     a parcel. Rooftop: The geocode point was matched to the rooftop of a building.
        [DataMember(Name = "calculationMethod", EmitDefaultValue = false)]
        public string CalculationMethod { get; set; }
        //
        // Summary:
        //     The best use for the geocode point. Can be Display or Route. Each geocode point
        //     is defined as a Route point, a Display point or both. Use Route points if you
        //     are creating a route to the location.Use Display points if you are showing the
        //     location on a map. For example, if the location is a park, a Route point may
        //     specify an entrance to the park where you can enter with a car, and a Display
        //     point may be a point that specifies the center of the park.
        [DataMember(Name = "usageTypes", EmitDefaultValue = false)]
        public string[] UsageTypes { get; set; }

        //
        // Summary:
        //     Converts the coordinate information in the point into a Coordinate object.
        //
        // Returns:
        //     Coordinate object representing the coordinate information in the point.
        public Coordinate GetCoordinate() { return new Coordinate(Coordinates[0], Coordinates[1]); }
    }

    [DataContract]
    [KnownType(typeof(Point))]
    public class Shape
    {
        public Shape() { }

        //
        // Summary:
        //     The bounding box of the shape.
        [DataMember(Name = "boundingBox", EmitDefaultValue = false)]
        public double[] BoundingBox { get; set; }
    }

    [DataContract]
    public class Coordinate
    {
        //
        // Summary:
        //     A location coordinate.
        public Coordinate() { }
        //
        // Summary:
        //     A location coordinate.
        //
        // Parameters:
        //   latitude:
        //     Latitude coordinate vlaue.
        //
        //   longitude:
        //     Longitude coordinate value.
        public Coordinate(double latitude, double longitude) { Latitude = latitude; Longitude = longitude; }

        //
        // Summary:
        //     Latitude coordinate.
        [DataMember(Name = "latitude")]
        public double Latitude { get; set; }
        //
        // Summary:
        //     Longitude coordinate.
        [DataMember(Name = "longitude")]
        public double Longitude { get; set; }

        //
        // Summary:
        //     Returns a formatted string of the coordinate in the format "latitude,longitude",
        //     with the values rounded off to 5 decimal places.
        public override string ToString() { return string.Format("{0}:{1}", Latitude, Longitude); }
    }

    [DataContract]
    public class RouteSubLeg
    {
        public RouteSubLeg() { }

        //
        // Summary:
        //     Information about the end waypoint of the route sub-leg.
        [DataMember(Name = "endWaypoint", EmitDefaultValue = false)]
        public Waypoint EndWaypoint { get; set; }
        //
        // Summary:
        //     Information about the start waypoint of the route sub-leg.
        [DataMember(Name = "startWaypoint", EmitDefaultValue = false)]
        public Waypoint StartWaypoint { get; set; }
        //
        // Summary:
        //     The physical distance covered by the sub-leg. The units are defined by the DistanceUnit
        //     field.
        [DataMember(Name = "travelDistance", EmitDefaultValue = false)]
        public double TravelDistance { get; set; }
        //
        // Summary:
        //     The time, in seconds, that it takes to travel the corresponding TravelDistance.
        [DataMember(Name = "travelDuration", EmitDefaultValue = false)]
        public double TravelDuration { get; set; }
    }

    [DataContract]
    public class Waypoint : Point
    {
        public Waypoint() { }

        //
        // Summary:
        //     A short description identifying the waypoint.
        [DataMember(Name = "description", EmitDefaultValue = false)]
        public string Description { get; set; }
        //
        // Summary:
        //     A value of true indicates that this is a via-waypoint.
        [DataMember(Name = "isVia", EmitDefaultValue = false)]
        public bool IsVia { get; set; }
        //
        // Summary:
        //     A unique identifier for the location.
        [DataMember(Name = "locationIdentifier", EmitDefaultValue = false)]
        public string LocationIdentifier { get; set; }
        //
        // Summary:
        //     Specifies the route path point associated with the waypoint.
        [DataMember(Name = "routePathIndex", EmitDefaultValue = false)]
        public int RoutePathIndex { get; set; }
    }

    [DataContract]
    public class ItineraryItem
    {
        public ItineraryItem() { }

        //
        // Summary:
        //     The time that it takes, in seconds, to travel a corresponding TravelDistance.
        [DataMember(Name = "travelDuration", EmitDefaultValue = false)]
        public double TravelDuration { get; set; }
        //
        // Summary:
        //     The physical distance covered by this route step. This value is not supported
        //     for the Transit travel mode.
        [DataMember(Name = "travelDistance", EmitDefaultValue = false)]
        public double TravelDistance { get; set; }
        //
        // Summary:
        //     The end destination for the transit line in the direction you are traveling.
        [DataMember(Name = "transitTerminus", EmitDefaultValue = false)]
        public string TransitTerminus { get; set; }
        //
        // Summary:
        //     The ID assigned to the transit stop by the transit agency.
        [DataMember(Name = "transitStopId", EmitDefaultValue = false)]
        public int TransitStopId { get; set; }
        //
        // Summary:
        //     Information about the transit line associated with the itinerary item.
        [DataMember(Name = "transitLine", EmitDefaultValue = false)]
        public TransitLine TransitLine { get; set; }
        //
        // Summary:
        //     The name of the street that the route goes towards in the first itinerary item.
        [DataMember(Name = "towardsRoadName", EmitDefaultValue = false)]
        public string TowardsRoadName { get; set; }
        //
        // Summary:
        //     The name or number of the toll zone.
        [DataMember(Name = "tollZone", EmitDefaultValue = false)]
        public string TollZone { get; set; }
        //
        // Summary:
        //     The arrival or departure time for the transit step.
        public DateTime TimeUtc { get; set; }
        //
        // Summary:
        //     The arrival or departure time for the transit step.
        [DataMember(Name = "time", EmitDefaultValue = false)]
        public string Time { get; set; }
        //
        // Summary:
        //     Signage text for the route. There may be more than one sign value for an itinerary
        //     item.
        [DataMember(Name = "signs", EmitDefaultValue = false)]
        public string[] Signs { get; set; }
        //
        // Summary:
        //     The side of the street where the destination is found based on the arrival direction.
        //     This field applies to the last itinerary item only. Possible values include:
        //     Left, Right, Unknown
        [DataMember(Name = "sideOfStreet", EmitDefaultValue = false)]
        public string SideOfStreet { get; set; }
        //
        // Summary:
        //     The coordinates of a point on the Earth where a maneuver is required, such as
        //     a left turn. A ManeuverPoint contains Latitude and Longitude elements. This value
        //     is not supported for ItineraryItems that are part of a ChildItineraryItems collection.
        [DataMember(Name = "maneuverPoint", EmitDefaultValue = false)]
        public Point ManeuverPoint { get; set; }
        //
        // Summary:
        //     A description of a maneuver in a set of directions.
        [DataMember(Name = "instruction", EmitDefaultValue = false)]
        public Instruction Instruction { get; set; }
        //
        // Summary:
        //     The type of icon to display. Possible values include: None, Airline, Auto, Bus,
        //     Ferry, Train, Walk, Other
        [DataMember(Name = "iconType", EmitDefaultValue = false)]
        public string IconType { get; set; }
        //
        // Summary:
        //     Additional information that may be helpful in following a route. In addition
        //     to the hint text, this element has an attribute hintType that specifies what
        //     the hint refers to, such as “NextIntersection.” Hint is an optional element and
        //     a route step can contain more than one hint.
        [DataMember(Name = "hints", EmitDefaultValue = false)]
        public Hint[] Hints { get; set; }
        //
        // Summary:
        //     The name or number of the exit associated with this itinerary step.
        [DataMember(Name = "exit", EmitDefaultValue = false)]
        public string Exit { get; set; }
        //
        // Summary:
        //     Information about one of the maneuvers that is part of the itinerary item. An
        //     ItineraryItem can contain more than one Detail collection.
        [DataMember(Name = "details", EmitDefaultValue = false)]
        public Detail[] Details { get; set; }
        //
        // Summary:
        //     The direction of travel associated with a maneuver on a route, such as south
        //     or southwest. This value is not supported for the Transit travel mode.
        [DataMember(Name = "compassDirection", EmitDefaultValue = false)]
        public string CompassDirection { get; set; }
        //
        // Summary:
        //     A collection of ItineraryItems that divides an itinerary item into smaller steps.
        //     An itinerary item can have only one set of ChildItineraryItems.
        [DataMember(Name = "childItineraryItems", EmitDefaultValue = false)]
        public ItineraryItem[] ChildItineraryItems { get; set; }
        //
        // Summary:
        //     The mode of travel for a specific step in the route. This value is not supported
        //     for ItineraryItems that are part of a ChildItineraryItems collection.
        [DataMember(Name = "travelMode", EmitDefaultValue = false)]
        public string TravelMode { get; set; }
        //
        // Summary:
        //     Information about a condition that may affect a specific step in the route. Warning
        //     is an optional element and a route step can contain more than one warning. Warnings
        //     can include traffic incident information such as congestion, accident and blocked
        //     road reports. Warning elements are further defined by two attributes: Severity
        //     and WarningType. Severity can have the following values: Low Impact, Minor, Moderate,
        //     or Serious.
        [DataMember(Name = "warnings", EmitDefaultValue = false)]
        public Warning[] Warnings { get; set; }
    }

    [DataContract]
    public class TransitLine
    {
        public TransitLine() { }

        //
        // Summary:
        //     The full name of the transit line.
        [DataMember(Name = "verboseName", EmitDefaultValue = false)]
        public string VerboseName { get; set; }
        //
        // Summary:
        //     The abbreviated name of the transit line, such as the bus number.
        [DataMember(Name = "abbreviatedName", EmitDefaultValue = false)]
        public string AbbreviatedName { get; set; }
        //
        // Summary:
        //     The ID associated with the transit agency.
        [DataMember(Name = "agencyId", EmitDefaultValue = false)]
        public long AgencyId { get; set; }
        //
        // Summary:
        //     The name of the transit agency.
        [DataMember(Name = "agencyName", EmitDefaultValue = false)]
        public string AgencyName { get; set; }
        //
        // Summary:
        //     The color associated with the transit line. The color is provided as an RGB value.
        [DataMember(Name = "lineColor", EmitDefaultValue = false)]
        public long LineColor { get; set; }
        //
        // Summary:
        //     The color to use for text associated with the transit line. The color is provided
        //     as an RGB value.
        [DataMember(Name = "lineTextColor", EmitDefaultValue = false)]
        public long LineTextColor { get; set; }
        //
        // Summary:
        //     The URI for the transit agency.
        [DataMember(Name = "uri", EmitDefaultValue = false)]
        public string Uri { get; set; }
        //
        // Summary:
        //     The phone number of the transit agency.
        [DataMember(Name = "phoneNumber", EmitDefaultValue = false)]
        public string PhoneNumber { get; set; }
        //
        // Summary:
        //     The contact information for the provider of the transit information.
        [DataMember(Name = "providerInfo", EmitDefaultValue = false)]
        public string ProviderInfo { get; set; }
    }

    [DataContract]
    public class Instruction
    {
        public Instruction() { }

        //
        // Summary:
        //     The textual representation of the maneuver.
        [DataMember(Name = "maneuverType", EmitDefaultValue = false)]
        public string ManeuverType { get; set; }
        //
        // Summary:
        //     The plain text description of the instruction.
        [DataMember(Name = "text", EmitDefaultValue = false)]
        public string Text { get; set; }
        //
        // Summary:
        //     The formatted text description of the instruction.
        [DataMember(Name = "formattedText", EmitDefaultValue = false)]
        public string FormattedText { get; set; }
    }

    [DataContract]
    public class Hint
    {
        public Hint() { }

        //
        // Summary:
        //     The type of hint.
        [DataMember(Name = "hintType", EmitDefaultValue = false)]
        public string HintType { get; set; }
        //
        // Summary:
        //     The hint information.
        [DataMember(Name = "text", EmitDefaultValue = false)]
        public string Text { get; set; }
    }

    [DataContract]
    public class Detail
    {
        public Detail() { }

        //
        // Summary:
        //     The direction in degrees. This value is not supported for the Transit travel
        //     mode.
        [DataMember(Name = "compassDegrees", EmitDefaultValue = false)]
        public int CompassDegrees { get; set; }
        //
        // Summary:
        //     The type of maneuver described by this detail collection. The ManeuverType in
        //     A detail collection can provide information for a portion of the maneuver described
        //     by the maneuverType attribute of the corresponding Instruction.
        [DataMember(Name = "maneuverType", EmitDefaultValue = false)]
        public string ManeuverType { get; set; }
        //
        // Summary:
        //     An enumeration version of the ManeuverType.
        public ManeuverType ManeuverTypeEnum { get; set; }
        //
        // Summary:
        //     This field specifies the index values for specific route path points that are
        //     returned in the response when a route path is returned. This and the EndPathIndices
        //     correspond to a maneuver route path.
        [DataMember(Name = "startPathIndices", EmitDefaultValue = false)]
        public int[] StartPathIndices { get; set; }
        //
        // Summary:
        //     This field specifies the index values for specific route path points that are
        //     returned in the response when a route path is returned. This and the StartPathIndices
        //     correspond to a maneuver route path.
        [DataMember(Name = "endPathIndices", EmitDefaultValue = false)]
        public int[] EndPathIndices { get; set; }
        //
        // Summary:
        //     The type of road.
        [DataMember(Name = "roadType", EmitDefaultValue = false)]
        public string RoadType { get; set; }
        //
        // Summary:
        //     A traffic location code. Each location code provides traffic incident information
        //     for pre-defined road segments. There may be multiple codes for each Detail collection
        //     in the response. A subscription is typically required to be able to interpret
        //     these codes for a geographical area or country.
        [DataMember(Name = "locationCodes", EmitDefaultValue = false)]
        public string[] LocationCodes { get; set; }
        //
        // Summary:
        //     A street, highway or intersection where the maneuver occurs. If the maneuver
        //     is complex, there may be more than one name field in the details collection.
        //     The name field may also have no value. This can occur if the name is not known
        //     or if a street, highway or intersection does not have a name. This value is only
        //     supported for the transit travel mode.
        [DataMember(Name = "names", EmitDefaultValue = false)]
        public string[] Names { get; set; }
        //
        // Summary:
        //     Describes the mode of transportation used between a pair of indexes. This can
        //     differ depending whether the route requires walking, driving, or transit. Not
        //     all regions or cultures support all values of this field.
        [DataMember(Name = "mode", EmitDefaultValue = false)]
        public string Mode { get; set; }
        //
        // Summary:
        //     Information on the road shield.
        [DataMember(Name = "roadShieldRequestParameters", EmitDefaultValue = false)]
        public RoadShield RoadShieldRequestParameters { get; set; }
    }

    public enum ManeuverType
    {
        //
        // Summary:
        //     Arrive at the final destination.
        ArriveFinish = 0,
        //
        // Summary:
        //     Arrive at an intermediate waypoint.
        ArriveIntermediate = 1,
        //
        // Summary:
        //     Bear left.
        BearLeft = 2,
        //
        // Summary:
        //     Bear left and then bear left again.
        BearLeftThenBearLeft = 3,
        //
        // Summary:
        //     Bear left and then bear right.
        BearLeftThenBearRight = 4,
        //
        // Summary:
        //     Bear left and then turn left.
        BearLeftThenTurnLeft = 5,
        //
        // Summary:
        //     Bear left and then turn right.
        BearLeftThenTurnRight = 6,
        //
        // Summary:
        //     Bear right.
        BearRight = 7,
        //
        // Summary:
        //     Bear right and then bear left.
        BearRightThenBearLeft = 8,
        //
        // Summary:
        //     Bear right and then bear right again.
        BearRightThenBearRight = 9,
        //
        // Summary:
        //     Bear right and then turn left.
        BearRightThenTurnLeft = 10,
        //
        // Summary:
        //     Bear right and then turn right.
        BearRightThenTurnRight = 11,
        //
        // Summary:
        //     Bear instruction and then a keep instruction
        BearThenKeep = 12,
        //
        // Summary:
        //     Bear instruction and then a merge instruction.
        BearThenMerge = 13,
        //
        // Summary:
        //     Continue on the current road.
        Continue = 14,
        //
        // Summary:
        //     Leave an intermediate waypoint in a different direction and road than you arrived
        //     on.
        DepartIntermediateStop = 15,
        //
        // Summary:
        //     Leave an intermediate waypoint in the same direction and on the same road that
        //     you arrived on.
        DepartIntermediateStopReturning = 16,
        //
        // Summary:
        //     Leave the starting point.
        DepartStart = 17,
        //
        // Summary:
        //     Enter a roundabout.
        EnterRoundabout = 18,
        //
        // Summary:
        //     Exit a roundabout.
        ExitRoundabout = 19,
        //
        // Summary:
        //     Enter and exit a roundabout.
        EnterThenExitRoundabout = 20,
        //
        // Summary:
        //     Keep left onto a different road.
        KeepLeft = 21,
        //
        // Summary:
        //     Keep left and continue onto ramp.
        KeepOnRampLeft = 22,
        //
        // Summary:
        //     Keep right and continue onto ramp.
        KeepOnRampRight = 23,
        //
        // Summary:
        //     Keep straight and continue onto ramp.
        KeepOnRampStraight = 24,
        //
        // Summary:
        //     Keep right onto a different road.
        KeepRight = 25,
        //
        // Summary:
        //     Keep straight onto a different road.
        KeepStraight = 26,
        //
        // Summary:
        //     Keep left to stay on the same road.
        KeepToStayLeft = 27,
        //
        // Summary:
        //     Keep right to stay on the same road.
        KeepToStayRight = 28,
        //
        // Summary:
        //     Keep straight to stay on the same road.
        KeepToStayStraight = 29,
        //
        // Summary:
        //     Merge onto a highway.
        Merge = 30,
        //
        // Summary:
        //     No instruction.
        None = 31,
        //
        // Summary:
        //     Take left ramp onto highway. This is part of a combined instruction.
        RampThenHighwayLeft = 32,
        //
        // Summary:
        //     Take right ramp onto highway. This is part of a combined instruction.
        RampThenHighwayRight = 33,
        //
        // Summary:
        //     Stay straight to take ramp onto highway. This is part of a combined instruction.
        RampThenHighwayStraight = 34,
        //
        // Summary:
        //     Road name changes.
        RoadNameChange = 35,
        //
        // Summary:
        //     Take the road. This instruction is used when you are entering or exiting a ferry.
        Take = 36,
        //
        // Summary:
        //     Take ramp to the left.
        TakeRampLeft = 37,
        //
        // Summary:
        //     Take ramp to the right.
        TakeRampRight = 38,
        //
        // Summary:
        //     Stay straight to take ramp.
        TakeRampStraight = 39,
        //
        // Summary:
        //     Take transit.
        TakeTransit = 40,
        //
        // Summary:
        //     Transfer between public transit at transit stop.
        Transfer = 41,
        //
        // Summary:
        //     Get off public transit at transit stop.
        TransitArrive = 42,
        //
        // Summary:
        //     Get on public transit at transit stop.
        TransitDepart = 43,
        //
        // Summary:
        //     Turn back sharply.
        TurnBack = 44,
        //
        // Summary:
        //     Turn left.
        TurnLeft = 45,
        //
        // Summary:
        //     Turn left and then bear left.
        TurnLeftThenBearLeft = 46,
        //
        // Summary:
        //     Turn left and then bear right.
        TurnLeftThenBearRight = 47,
        //
        // Summary:
        //     Turn left and then turn left again.
        TurnLeftThenTurnLeft = 48,
        //
        // Summary:
        //     Turn left and then turn right.
        TurnLeftThenTurnRight = 49,
        //
        // Summary:
        //     Turn right.
        TurnRight = 50,
        //
        // Summary:
        //     Turn right and then bear left.
        TurnRightThenBearLeft = 51,
        //
        // Summary:
        //     Turn right and then bear right.
        TurnRightThenBearRight = 52,
        //
        // Summary:
        //     Turn right and then turn left.
        TurnRightThenTurnLeft = 53,
        //
        // Summary:
        //     Turn right and then turn right again
        TurnRightThenTurnRight = 54,
        //
        // Summary:
        //     Turn instruction followed by a merge instruction.
        TurnThenMerge = 55,
        //
        // Summary:
        //     Turn left to stay on the same road.
        TurnToStayLeft = 56,
        //
        // Summary:
        //     Turn right to stay on the same road.
        TurnToStayRight = 57,
        //
        // Summary:
        //     The instruction is unknown.
        Unknown = 58,
        //
        // Summary:
        //     Make a u-turn to go in the opposite direction.
        UTurn = 59,
        //
        // Summary:
        //     Wait at a transit stop.
        Wait = 60,
        //
        // Summary:
        //     Walk.
        Walk = 61
    }

    [DataContract]
    public class RoadShield
    {
        public RoadShield() { }

        //
        // Summary:
        //     The bucket id to used to display the shield.
        [DataMember(Name = "bucket", EmitDefaultValue = false)]
        public int Bucket { get; set; }
        //
        // Summary:
        //     The shield details.
        [DataMember(Name = "shields", EmitDefaultValue = false)]
        public Shield[] Shields { get; set; }
    }

    [DataContract]
    public class Shield
    {
        public Shield() { }

        //
        // Summary:
        //     The labels to add to the shield.
        [DataMember(Name = "labels", EmitDefaultValue = false)]
        public string[] Labels { get; set; }
        //
        // Summary:
        //     The type of shield.
        [DataMember(Name = "roadShieldType", EmitDefaultValue = false)]
        public int RoadShieldType { get; set; }
    }

    [DataContract]
    public class Warning
    {
        public Warning() { }

        //
        // Summary:
        //     Where the warning starts.
        [DataMember(Name = "origin", EmitDefaultValue = false)]
        public string Origin { get; set; }
        //
        // Summary:
        //     Where the warning starts.
        public Coordinate OriginCoordinate { get; set; }
        //
        // Summary:
        //     Severity can have the following values: Low Impact, Minor, Moderate, or Serious.
        [DataMember(Name = "severity", EmitDefaultValue = false)]
        public string Severity { get; set; }
        //
        // Summary:
        //     The warning information.
        [DataMember(Name = "text", EmitDefaultValue = false)]
        public string Text { get; set; }
        //
        // Summary:
        //     Where the warning ends.
        [DataMember(Name = "to", EmitDefaultValue = false)]
        public string To { get; set; }
        //
        // Summary:
        //     Where the warning ends.
        public Coordinate ToCoordinate { get; set; }
        //
        // Summary:
        //     The type of warning.
        [DataMember(Name = "warningType", EmitDefaultValue = false)]
        public string WarningType { get; set; }
        //
        // Summary:
        //     An enumeration version of the warning type.
        public WarningType WarningTypeEnum { get; set; }
    }

    public enum WarningType
    {
        //
        // Summary:
        //     There is a traffic accident.
        Accident = 0,
        //
        // Summary:
        //     The route has left one administrative division and entered another.
        AdminDivisionChange = 1,
        //
        // Summary:
        //     The road is closed or blocked.
        BlockedRoad = 2,
        //
        // Summary:
        //     Check a time table. This usually refers to a ferry or Autorail time table.
        CheckTimetable = 3,
        //
        // Summary:
        //     The traffic is slow.
        Congestion = 4,
        //
        // Summary:
        //     The route has left one country and entered another.
        CountryChange = 5,
        //
        // Summary:
        //     There is a disabled vehicle.
        DisabledVehicle = 6,
        //
        // Summary:
        //     A gate blocks the road and access is required to continue along the route.
        GateAccess = 7,
        //
        // Summary:
        //     Get off the transit at this location.
        GetOffTransit = 8,
        //
        // Summary:
        //     Get on the transit at this location.
        GetOnTransit = 9,
        //
        // Summary:
        //     A U-turn is illegal at this location.
        IllegalUTurn = 10,
        //
        // Summary:
        //     There is mass transit incident.
        MassTransit = 11,
        //
        // Summary:
        //     A miscellaneous warning is available for this location.
        Miscellaneous = 12,
        //
        // Summary:
        //     There is no incident at this location.
        NoIncident = 13,
        //
        // Summary:
        //     There is no warning at this location.
        None = 14,
        //
        // Summary:
        //     There is a warning at this location that cannot be classified as any other type
        //     of warning.
        Other = 15,
        //
        // Summary:
        //     There is additional traffic incident information.
        OtherNews = 16,
        //
        // Summary:
        //     There are other traffic incidents at this location.
        OtherTrafficIncidents = 17,
        //
        // Summary:
        //     There are scheduled events in the area.
        PlannedEvents = 18,
        //
        // Summary:
        //     The road being travelled on is private.
        PrivateRoad = 19,
        //
        // Summary:
        //     The turn may be restricted.
        RestrictedTurn = 20,
        //
        // Summary:
        //     There are road closures at this location.
        RoadClosures = 21,
        //
        // Summary:
        //     There is a road hazard.
        RoadHazard = 22,
        //
        // Summary:
        //     There is construction along the route. The ScheduledConstruction value is used
        //     for any type of construction and not just construction that has specific start
        //     and end dates.
        ScheduledConstruction = 23,
        //
        // Summary:
        //     A seasonal closure occurs at this location.
        SeasonalClosures = 24,
        //
        // Summary:
        //     A toll is required at this location to continue along the route.
        Tollbooth = 25,
        //
        // Summary:
        //     The road is a toll road.
        TollRoad = 26,
        //
        // Summary:
        //     The entrance to a toll zone.
        TollZoneEnter = 27,
        //
        // Summary:
        //     The exit of a toll zone.
        TollZoneExit = 28,
        //
        // Summary:
        //     The warning is about traffic flow.
        TrafficFlow = 29,
        //
        // Summary:
        //     There is a transit line change but a change of vehicle is not required.
        TransitLineChange = 30,
        //
        // Summary:
        //     The road is unpaved.
        UnpavedRoad = 31,
        //
        // Summary:
        //     There is significant weather at this location.
        Weather = 32
    }

    //
    // Summary:
    //     A traffic incident response object.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class TrafficIncident : Resource
    {
        public TrafficIncident() { }

        //
        // Summary:
        //     A collection of traffic location codes. This field is provided when you set the
        //     includeLocationCodes parameter to true in the request. These codes associate
        //     an incident with pre-defined road segments. A subscription is typically required
        //     to be able to interpret these codes for a geographical area or country.
        [DataMember(Name = "locationCodes", EmitDefaultValue = false)]
        public string[] LocationCodes { get; set; }
        //
        // Summary:
        //     The coordinates of the end of a traffic incident, such as the end of a construction
        //     zone.
        [DataMember(Name = "toPoint", EmitDefaultValue = false)]
        public Point ToPoint { get; set; }
        //
        // Summary:
        //     Specifies the level of importance of incident. Where: 1: LowImpact, 2: Minor,
        //     3: Moderate, 4: Serious.
        [DataMember(Name = "severity", EmitDefaultValue = false)]
        public int Severity { get; set; }
        //
        // Summary:
        //     A value of true indicates that there is a road closure.
        [DataMember(Name = "roadClosed", EmitDefaultValue = false)]
        public bool RoadClosed { get; set; }
        //
        // Summary:
        //     The time the incident information was last updated as a DateTime Object.
        public DateTime LastModifiedDateTimeUtc { get; set; }
        //
        // Summary:
        //     The time the incident information was last updated as an Odata JSON Date string.
        [DataMember(Name = "lastModified", EmitDefaultValue = false)]
        public string LastModified { get; set; }
        //
        // Summary:
        //     A description specific to lanes, such as lane closures.
        [DataMember(Name = "lane", EmitDefaultValue = false)]
        public string Lane { get; set; }
        //
        // Summary:
        //     Specifies the type of incident. Where 1: Accident, 2: Congestion, 3: DisabledVehicle,
        //     4: MassTransit, 5: Miscellaneous, 6: OtherNews, 7: PlannedEvent, 8: RoadHazard,
        //     9: Construction, 10: Alert, 11: Weather
        [DataMember(Name = "type", EmitDefaultValue = false)]
        public new int Type { get; set; }
        //
        // Summary:
        //     A unique ID for the incident.
        [DataMember(Name = "incidentId", EmitDefaultValue = false)]
        public long IncidentId { get; set; }
        //
        // Summary:
        //     The time the incident will end as an Odata JSON Date string.
        [DataMember(Name = "end", EmitDefaultValue = false)]
        public string End { get; set; }
        //
        // Summary:
        //     The time the incident occurred as a DateTime Object.
        public DateTime StartDateTimeUtc { get; set; }
        //
        // Summary:
        //     The time the incident occurred as an Odata JSON Date string.
        [DataMember(Name = "start", EmitDefaultValue = false)]
        public string Start { get; set; }
        //
        // Summary:
        //     A description of a detour.
        [DataMember(Name = "detour", EmitDefaultValue = false)]
        public string Detour { get; set; }
        //
        // Summary:
        //     A description of the incident.
        [DataMember(Name = "description", EmitDefaultValue = false)]
        public string Description { get; set; }
        //
        // Summary:
        //     A description of the congestion.
        [DataMember(Name = "congestion", EmitDefaultValue = false)]
        public string Congestion { get; set; }
        //
        // Summary:
        //     The latitude and longitude coordinates where you encounter the incident.
        [DataMember(Name = "point", EmitDefaultValue = false)]
        public Point Point { get; set; }
        //
        // Summary:
        //     The time the incident will end as a DateTime Object.
        public DateTime EndDateTimeUtc { get; set; }
        //
        // Summary:
        //     A value of true indicates that the incident has been visually verified or otherwise
        //     officially confirmed by a source like the local police department.
        [DataMember(Name = "verified", EmitDefaultValue = false)]
        public bool Verified { get; set; }
    }

    //
    // Summary:
    //     A Imagery metadata response object.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    [KnownType(typeof(StaticMapMetadata))]
    [KnownType(typeof(BirdseyeMetadata))]
    public class ImageryMetadata : Resource
    {
        public ImageryMetadata() { }

        //
        // Summary:
        //     The height of the image tile.
        [DataMember(Name = "imageHeight", EmitDefaultValue = false)]
        public int ImageHeight { get; set; }
        //
        // Summary:
        //     The width of the image tile.
        [DataMember(Name = "imageWidth", EmitDefaultValue = false)]
        public int ImageWidth { get; set; }
        //
        // Summary:
        //     Information about the data providers of the imagery.
        [DataMember(Name = "imageryProviders", EmitDefaultValue = false)]
        public ImageryProvider[] ImageryProviders { get; set; }
        //
        // Summary:
        //     A URL template for an image tile if a specific point is specified or a general
        //     URL template for the specified imagery set depending on the request.
        [DataMember(Name = "imageUrl", EmitDefaultValue = false)]
        public string ImageUrl { get; set; }
        //
        // Summary:
        //     One or more URL subdomains that may be used when constructing an image tile URL.
        [DataMember(Name = "imageUrlSubdomains", EmitDefaultValue = false)]
        public string[] ImageUrlSubdomains { get; set; }
        //
        // Summary:
        //     The latest date found in an imagery set or for a specific imagery tile.
        [DataMember(Name = "vintageEnd", EmitDefaultValue = false)]
        public string VintageEnd { get; set; }
        //
        // Summary:
        //     The earliest date found in an imagery set or for a specific imagery tile.
        [DataMember(Name = "vintageStart", EmitDefaultValue = false)]
        public string VintageStart { get; set; }
        //
        // Summary:
        //     The maximum zoom level available for this imagery set.
        [DataMember(Name = "zoomMax", EmitDefaultValue = false)]
        public int ZoomMax { get; set; }
        //
        // Summary:
        //     The minimum zoom level available for this imagery set.
        [DataMember(Name = "zoomMin", EmitDefaultValue = false)]
        public int ZoomMin { get; set; }
    }

    [DataContract]
    public class ImageryProvider
    {
        public ImageryProvider() { }

        //
        // Summary:
        //     The attribution to be displayed when showing the imagery.
        [DataMember(Name = "attribution", EmitDefaultValue = false)]
        public string Attribution { get; set; }
        //
        // Summary:
        //     The coverage area of a data providers imagery.
        [DataMember(Name = "coverageAreas", EmitDefaultValue = false)]
        public CoverageArea[] CoverageAreas { get; set; }
    }

    [DataContract]
    public class CoverageArea
    {
        public CoverageArea() { }

        //
        // Summary:
        //     Bounding box of the coverage area. Structure [South Latitude, West Longitude,
        //     North Latitude, East Longitude]
        [DataMember(Name = "bbox", EmitDefaultValue = false)]
        public double[] BoundingBox { get; set; }
        //
        // Summary:
        //     The maximum zoom level available for this data set.
        [DataMember(Name = "zoomMax", EmitDefaultValue = false)]
        public int ZoomMax { get; set; }
        //
        // Summary:
        //     The minimum zoom level available for this data set.
        [DataMember(Name = "zoomMin", EmitDefaultValue = false)]
        public int ZoomMin { get; set; }
    }

    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class StaticMapMetadata : ImageryMetadata
    {
        public StaticMapMetadata() { }

        //
        // Summary:
        //     The center coordinate of the map image.
        [DataMember(Name = "mapCenter", EmitDefaultValue = false)]
        public Point MapCenter { get; set; }
        //
        // Summary:
        //     Information about pushpins in the image.
        [DataMember(Name = "pushpins", EmitDefaultValue = false)]
        public PushpinMetdata[] Pushpins { get; set; }
        //
        // Summary:
        //     The zoom level of the image.
        [DataMember(Name = "zoom", EmitDefaultValue = false)]
        public string Zoom { get; set; }
    }

    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class PushpinMetdata
    {
        public PushpinMetdata() { }

        //
        // Summary:
        //     The placement of the pushpin on the static map that is defined by an offset in
        //     pixels from the upper left hand corner of the map.
        [DataMember(Name = "anchor", EmitDefaultValue = false)]
        public Pixel Anchor { get; set; }
        //
        // Summary:
        //     The offset of the bottom right corner of the pushpin icon with respect to the
        //     anchor point.
        [DataMember(Name = "bottomRightOffset", EmitDefaultValue = false)]
        public Pixel BottomRightOffset { get; set; }
        //
        // Summary:
        //     The offset of the top left corner of the pushpin icon with respect to the anchor
        //     point.
        [DataMember(Name = "topLeftOffset", EmitDefaultValue = false)]
        public Pixel TopLeftOffset { get; set; }
        //
        // Summary:
        //     The latitude and longitude coordinates of the pushpin.
        [DataMember(Name = "point", EmitDefaultValue = false)]
        public Point Point { get; set; }
    }

    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class Pixel
    {
        public Pixel() { }

        //
        // Summary:
        //     The horizontal position.
        [DataMember(Name = "x", EmitDefaultValue = false)]
        public string X { get; set; }
        //
        // Summary:
        //     The vertical position.
        [DataMember(Name = "y", EmitDefaultValue = false)]
        public string Y { get; set; }
    }

    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class BirdseyeMetadata : ImageryMetadata
    {
        public BirdseyeMetadata() { }

        //
        // Summary:
        //     The orientation of the viewport for the imagery metadata in degrees where 0 =
        //     North, 90 = East, 180 = South, 270 = West.
        [DataMember(Name = "orientation", EmitDefaultValue = false)]
        public double Orientation { get; set; }
        //
        // Summary:
        //     The horizontal dimension of the imagery in number of tiles.
        [DataMember(Name = "tilesX", EmitDefaultValue = false)]
        public int TilesX { get; set; }
        //
        // Summary:
        //     The vertical dimension of the imagery in number of tiles.
        [DataMember(Name = "tilesY", EmitDefaultValue = false)]
        public int TilesY { get; set; }
    }

    //
    // Summary:
    //     An Elevation service response object.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class ElevationData : Resource
    {
        public ElevationData() { }

        //
        // Summary:
        //     A list of elevation values in meters.
        [DataMember(Name = "elevations", EmitDefaultValue = false)]
        public int[] Elevations { get; set; }
        //
        // Summary:
        //     The zoom level used to compute the elevation values. Zoom level values are from
        //     1 to 21. A lower value typically means less accurate results because the resolution
        //     of the elevation data is less. At lower resolutions, interpolated elevation values
        //     use data points that are further apart. The zoom level used depends on the amount
        //     of elevation data available in a region and can be lowered when the specified
        //     path covers a long distance.
        [DataMember(Name = "zoomLevel", EmitDefaultValue = false)]
        public int ZoomLevel { get; set; }
    }

    //
    // Summary:
    //     An Elevation service response object for Sea Level offset data.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class SeaLevelData : Resource
    {
        public SeaLevelData() { }

        //
        // Summary:
        //     The difference between sea level models for a set of locations.
        [DataMember(Name = "offsets", EmitDefaultValue = false)]
        public int[] Offsets { get; set; }
        //
        // Summary:
        //     The zoom level used to compute the elevation values. Zoom level values are from
        //     1 to 21. A lower value typically means less accurate results because the resolution
        //     of the elevation data is less. At lower resolutions, interpolated elevation values
        //     use data points that are further apart. The zoom level used depends on the amount
        //     of elevation data available in a region and can be lowered when the specified
        //     path covers a long distance.
        [DataMember(Name = "zoomLevel", EmitDefaultValue = false)]
        public int ZoomLevel { get; set; }
    }

    //
    // Summary:
    //     A compressed list of coordinates.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class CompressedPointList : Resource
    {
        public CompressedPointList() { }

        //
        // Summary:
        //     A string that represents a compressed list of coordinates.
        [DataMember(Name = "value", EmitDefaultValue = false)]
        public string Value { get; set; }
    }

    //
    // Summary:
    //     This response specifies: - Whether this is a politically disputed area, such
    //     as an area claimed by more than one country. - Whether services are available
    //     in the user’s region. - A list of available geospatial services including endpoints
    //     and language support for each service.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class GeospatialEndpointResponse : Resource
    {
        public GeospatialEndpointResponse() { }

        //
        // Summary:
        //     Specifies if this area in the request is claimed by more than one country.
        [DataMember(Name = "isDisputedArea", EmitDefaultValue = false)]
        public bool IsDisputedArea { get; set; }
        //
        // Summary:
        //     Specifies if Geospatial Platform services are available in the country or region.
        //     Microsoft does not support services in embargoed areas.
        [DataMember(Name = "isSupported", EmitDefaultValue = false)]
        public bool IsSupported { get; set; }
        //
        // Summary:
        //     The country or region that was used to determine service support. If you specified
        //     a User Location in the request that is in a non-disputed country or region, this
        //     country or region is returned in the response.
        [DataMember(Name = "ur", EmitDefaultValue = false)]
        public string UserRegion { get; set; }
        //
        // Summary:
        //     Information for each geospatial service that is available in the country or region
        //     and language specified in the request.
        [DataMember(Name = "services", EmitDefaultValue = false)]
        public GeospatialService[] Services { get; set; }
    }

    [DataContract]
    public class GeospatialService
    {
        public GeospatialService() { }

        //
        // Summary:
        //     The URL service endpoint to use in this region. Note that to use the service,
        //     you must typically add parameters specific to the service. These parameters are
        //     not described in this documentation.
        [DataMember(Name = "endpoint", EmitDefaultValue = false)]
        public string Endpoint { get; set; }
        //
        // Summary:
        //     Set to true if the service supports the language in the request for the region.
        //     If the language is supported, then the service endpoint will return responses
        //     using this language. If it is not supported, then the service will use the fallback
        //     language.
        [DataMember(Name = "fallbackLanguage", EmitDefaultValue = false)]
        public string FallbackLanguage { get; set; }
        //
        // Summary:
        //     Specifies the secondary or fallback language in this region or country. If the
        //     requested language is not supported and a fallback language is not available,
        //     United States English (en-us) is used.
        [DataMember(Name = "languageSupported", EmitDefaultValue = false)]
        public bool LanguageSupported { get; set; }
        //
        // Summary:
        //     An abbreviated name for the service.
        [DataMember(Name = "serviceName", EmitDefaultValue = false)]
        public string ServiceName { get; set; }
    }

    //
    // Summary:
    //     A Distance Matrix response object which is returned when geocoding or reverse
    //     geocoding.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class DistanceMatrix : Resource
    {
        public DistanceMatrix() { }

        //
        // Summary:
        //     A list of time intervals in which the distance matrix calculated for.
        public List<DateTime> TimeIntervals { get; set; }
        //
        // Summary:
        //     Details of an error that may have occurred when processing the request.
        [DataMember(Name = "errorMessage", EmitDefaultValue = false)]
        public string ErrorMessage { get; set; }
        //
        // Summary:
        //     The array of origins that were used to calculate the distance matrix.
        [DataMember(Name = "origins", EmitDefaultValue = false)]
        public SimpleWaypoint[] Origins { get; set; }
        //
        // Summary:
        //     The array of destinations that were used to calculate the distance matrix.
        [DataMember(Name = "destinations", EmitDefaultValue = false)]
        public SimpleWaypoint[] Destinations { get; set; }
        //
        // Summary:
        //     Array of distance matrix cell results containing information for each coordinate
        //     pair and time interval.
        [DataMember(Name = "results", EmitDefaultValue = false)]
        public DistanceMatrixCell[] Results { get; set; }

        //
        // Summary:
        //     Retrives the distance matrix cell for a specified origin-destination pair. Returns
        //     -1 if a cell can not be found in the results or had an error in calculation.
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        // Returns:
        //     The distance matrix cell for a specified origin-destination pair.
        public DistanceMatrixCell GetCell(int originIdx, int destinationIdx) { return null; }
        //
        // Summary:
        //     Retrives the distance matrix cell for a specified origin-destination pair and
        //     time interval. Returns -1 if a cell can not be found in the results or had an
        //     error in calculation.
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        //   timeIntervalIdx:
        //     The time interval to retrieve the distance matrix cell for.
        //
        // Returns:
        //     The distance matrix cell for a specified origin-destination pair and time interval.
        public DistanceMatrixCell GetCell(int originIdx, int destinationIdx, int timeIntervalIdx) { return null; }
        //
        // Summary:
        //     Retrives the distance matrix cell for a specified origin-destination pair and
        //     time interval. Returns -1 if a cell can not be found in the results or had an
        //     error in calculation.
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        //   timeInterval:
        //     The time interval to retrieve the distance matrix cell for.
        //
        // Returns:
        //     The distance matrix cell for a specified origin-destination pair and time interval.
        public DistanceMatrixCell GetCell(int originIdx, int destinationIdx, DateTime timeInterval) { return null; }
        //
        // Summary:
        //     Gets all cells for the specified origin and destination index, ordered by time
        //     (ascending).
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        // Returns:
        //     All cells for the specified origin and destination index, ordered by time.
        public DistanceMatrixCell[] GetCells(int originIdx, int destinationIdx) { return null; }
        //
        // Summary:
        //     Retrives the travel distance for a specified origin-destination pair and time
        //     interval. Returns -1 if a cell can not be found in the results or had an error
        //     in calculation.
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        //   timeInterval:
        //     The time interval to retrieve the travel distance for.
        //
        // Returns:
        //     The travel distance for a specified origin-destination pair and time interval.
        public double GetDistance(int originIdx, int destinationIdx, DateTime timeInterval) { return 0; }
        //
        // Summary:
        //     Retrives the travel distance for a specified origin-destination pair. Returns
        //     -1 if a cell can not be found in the results or had an error in calculation.
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        // Returns:
        //     The travel distance for a specified origin-destination pair.
        public double GetDistance(int originIdx, int destinationIdx) { return 0; }
        //
        // Summary:
        //     Retrives the travel distance for a specified origin-destination pair and time
        //     interval. Returns -1 if a cell can not be found in the results or had an error
        //     in calculation.
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        //   timeIntervalIdx:
        //     The time interval to retrieve the travel distance for.
        //
        // Returns:
        //     The travel distance for a specified origin-destination pair and time interval.
        public double GetDistance(int originIdx, int destinationIdx, int timeIntervalIdx) { return 0; }
        //
        // Summary:
        //     Gets all travel distances for the specified origin and destination index, ordered
        //     by time (ascending).
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        // Returns:
        //     All travel distances for the specified origin and destination index, ordered
        //     by time.
        public double[] GetDistances(int originIdx, int destinationIdx) { return null; }
        //
        // Summary:
        //     Retrieves the travel time for a specified origin-destination pair and time interval.
        //     Returns -1 if a cell can not be found in the results or had an error in calculation.
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        //   timeInterval:
        //     The time interval to retrieve the travel time for.
        //
        // Returns:
        //     The travel time for a specified origin-destination pair and time interval.
        public double GetTime(int originIdx, int destinationIdx, DateTime timeInterval) { return 0; }
        //
        // Summary:
        //     Retrieves the travel time for a specified origin-destination pair and time interval.
        //     Returns -1 if a cell can not be found in the results or had an error in calculation.
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        //   timeIntervalIdx:
        //     The index of the time interval.
        //
        // Returns:
        //     The travel time for a specified origin-destination pair and time interval.
        public double GetTime(int originIdx, int destinationIdx, int timeIntervalIdx) { return 0; }
        //
        // Summary:
        //     Retrives the travel time for a specified origin-destination pair. Returns -1
        //     if a cell can not be found in the results or had an error in calculation.
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        // Returns:
        //     The travel time for a specified origin-destination pair.
        public double GetTime(int originIdx, int destinationIdx) { return 0; }
        //
        // Summary:
        //     Gets all travel times for the specified origin and destination index, ordered
        //     by time (ascending).
        //
        // Parameters:
        //   originIdx:
        //     The index of the origin in the requests origins array.
        //
        //   destinationIdx:
        //     The index of the destination in the requests destinations array.
        //
        // Returns:
        //     All travel times for the specified origin and destination index, ordered by time.
        public double[] GetTimes(int originIdx, int destinationIdx) { return null; }
    }

    [DataContract]
    public class SimpleWaypoint
    {
        //
        // Summary:
        //     A waypoint to calculate a route through.
        public SimpleWaypoint() { }
        //
        // Summary:
        //     A waypoint to calculate a route through.
        //
        // Parameters:
        //   coordinate:
        //     Coordinate of the waypoint.
        public SimpleWaypoint(Coordinate coordinate) { }
        //
        // Summary:
        //     A waypoint to calculate a route through.
        //
        // Parameters:
        //   address:
        //     Address or location description of waypoint.
        public SimpleWaypoint(string address) { }
        //
        // Summary:
        //     A waypoint to calculate a route through.
        //
        // Parameters:
        //   coordinate:
        //     Coordinate of the waypoint.
        //
        //   address:
        //     Address or location description of waypoint.
        public SimpleWaypoint(Coordinate coordinate, string address) { }
        //
        // Summary:
        //     A waypoint to calculate a route through.
        //
        // Parameters:
        //   latitude:
        //     The latitude coordinate.
        //
        //   longitude:
        //     The longitude coordinate.
        public SimpleWaypoint(double latitude, double longitude) { }

        //
        // Summary:
        //     The coordinate of the waypoint. When specified this will be used instead of the
        //     Address value in requests.
        public Coordinate Coordinate { get; set; }
        //
        // Summary:
        //     The address query for the waypoint.
        public string Address { get; set; }
        //
        // Summary:
        //     A bool indicating whether the waypoint is a via point.
        public bool IsViaPoint { get; set; }
    }

    [DataContract]
    public class DistanceMatrixCell
    {
        public DistanceMatrixCell() { }

        //
        // Summary:
        //     The index of the origin in the provided origins array used to calculate this
        //     cell.
        [DataMember(Name = "originIndex", EmitDefaultValue = false)]
        public int OriginIndex { get; set; }
        //
        // Summary:
        //     The index of the destination in the provided destinations array used to calculate
        //     this cell.
        [DataMember(Name = "destinationIndex", EmitDefaultValue = false)]
        public int DestinationIndex { get; set; }
        //
        // Summary:
        //     The physical distance covered to complete a route between the origin and destination.
        [DataMember(Name = "travelDistance", EmitDefaultValue = false)]
        public double TravelDistance { get; set; }
        //
        // Summary:
        //     The time that it takes, in minutes, to travel a corresponding TravelDistance.
        [DataMember(Name = "travelDuration", EmitDefaultValue = false)]
        public double TravelDuration { get; set; }
        //
        // Summary:
        //     The departure time in which this cell was calculated for. Only returned when
        //     a startTime is specified. When an endTime is specified in the request several
        //     cells will be returned for the same origin and destination pairs, each having
        //     a different departure time for each time interval in the generated histogram
        //     request. This string is in ISO 8601 date format.
        [DataMember(Name = "departureTime", EmitDefaultValue = false)]
        public string DepartureTime { get; set; }
        //
        // Summary:
        //     The departure time in which this cell was calculated for. Only returned when
        //     a startTime is specified. When an endTime is specified in the request several
        //     cells will be returned for the same origin and destination pairs, each having
        //     a different departure time for each time interval in the generated histogram
        //     request.
        public DateTime? DepartureTimeUtc { get; set; }
        //
        // Summary:
        //     A boolean indicating if an error occurred when calculating this cell.
        [DataMember(Name = "hasError", EmitDefaultValue = false)]
        public bool HasError { get; set; }
    }

    //
    // Summary:
    //     The status of an asynchronous distance matrix request.
    [DataContract(Namespace = "http://schemas.microsoft.com/search/local/ws/rest/v1")]
    public class DistanceMatrixAsyncStatus : Resource
    {
        public DistanceMatrixAsyncStatus() { }

        //
        // Summary:
        //     Specifies if the request is accepted. A request will not be accepted if it is
        //     not valid, is not within the coordinate pair limits, or if a coordinate in origins
        //     or destinations has no coverage.
        [DataMember(Name = "isAccepted", EmitDefaultValue = false)]
        public bool IsAccepted { get; set; }
        //
        // Summary:
        //     Specifies if the request has completed.
        [DataMember(Name = "isCompleted", EmitDefaultValue = false)]
        public bool IsCompleted { get; set; }
        //
        // Summary:
        //     A unique identifier for an asynchronous request. This can be used to retrieve
        //     the results of an asynchronous request when it has completed.
        [DataMember(Name = "requestId", EmitDefaultValue = false)]
        public string RequestId { get; set; }
        //
        // Summary:
        //     Details of an error that may have occurred when processing the request.
        [DataMember(Name = "errorMessage", EmitDefaultValue = false)]
        public string ErrorMessage { get; set; }
        //
        // Summary:
        //     An estimated number of seconds to wait before calling back for results when making
        //     an asynchronous request.
        [DataMember(Name = "callbackInSeconds", EmitDefaultValue = false)]
        public int CallbackInSeconds { get; set; }
        //
        // Summary:
        //     The callback URL to use to check the status of the request.
        [DataMember(Name = "callbackUrl", EmitDefaultValue = false)]
        public string CallbackUrl { get; set; }
        //
        // Summary:
        //     The URL where the results can be downloaded from. The result is a DistanceMatrix
        //     object in JSON form.
        [DataMember(Name = "resultUrl", EmitDefaultValue = false)]
        public string ResultUrl { get; set; }
    }

    [DataContract]
    public class Address
    {
        public Address() { }

        //
        // Summary:
        //     The official street line of an address relative to the area, as specified by
        //     the Locality, or PostalCode, properties. Typical use of this element would be
        //     to provide a street address or any official address.
        [DataMember(Name = "addressLine", EmitDefaultValue = false)]
        public string AddressLine { get; set; }
        //
        // Summary:
        //     A string specifying the subdivision name in the country or region for an address.
        //     This element is typically treated as the first order administrative subdivision,
        //     but in some cases it is the second, third, or fourth order subdivision in a country,
        //     dependency, or region.
        [DataMember(Name = "adminDistrict", EmitDefaultValue = false)]
        public string AdminDistrict { get; set; }
        //
        // Summary:
        //     A string specifying the subdivision name in the country or region for an address.
        //     This element is used when there is another level of subdivision information for
        //     a location, such as the county.
        [DataMember(Name = "adminDistrict2", EmitDefaultValue = false)]
        public string AdminDistrict2 { get; set; }
        //
        // Summary:
        //     A string specifying the country or region name of an address.
        [DataMember(Name = "countryRegion", EmitDefaultValue = false)]
        public string CountryRegion { get; set; }
        //
        // Summary:
        //     A string specifying the populated place for the address. This typically refers
        //     to a city, but may refer to a suburb or a neighborhood in certain countries.
        [DataMember(Name = "locality", EmitDefaultValue = false)]
        public string Locality { get; set; }
        //
        // Summary:
        //     A string specifying the post code, postal code, or ZIP Code of an address.
        [DataMember(Name = "postalCode", EmitDefaultValue = false)]
        public string PostalCode { get; set; }
        //
        // Summary:
        //     A string specifying the two-letter ISO country code. You must specify include=ciso2
        //     in your request to return this ISO country code.
        [DataMember(Name = "countryRegionIso2", EmitDefaultValue = false)]
        public string CountryRegionIso2 { get; set; }
        //
        // Summary:
        //     A string specifying the complete address. This address may not include the country
        //     or region.
        [DataMember(Name = "formattedAddress", EmitDefaultValue = false)]
        public string FormattedAddress { get; set; }
        //
        // Summary:
        //     A string specifying the neighborhood for an address. You must specify includeNeighborhood=1
        //     in your request to return the neighborhood.
        [DataMember(Name = "neighborhood", EmitDefaultValue = false)]
        public string Neighborhood { get; set; }
        //
        // Summary:
        //     A string specifying the name of the landmark when there is a landmark associated
        //     with an address.
        [DataMember(Name = "landmark", EmitDefaultValue = false)]
        public string Landmark { get; set; }
    }

    [DataContract]
    public class QueryParseValue
    {
        public QueryParseValue() { }

        //
        // Summary:
        //     The name of the property parsed. Can be AddressLine, Locality, AdminDistrict,
        //     AdminDistrict2, PostalCode, CountryRegion, Landmark.
        [DataMember(Name = "property", EmitDefaultValue = false)]
        public string Property { get; set; }
        //
        // Summary:
        //     The parsed value.
        [DataMember(Name = "value", EmitDefaultValue = false)]
        public string Value { get; set; }
    }
}
